<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Validation\Rule;

class CategoryController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:categories.read', only: ['index']),
            new Middleware('permission:categories.create', only: ['create', 'store']),
            new Middleware('permission:categories.update', only: ['edit','update']),
            new Middleware('permission:categories.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = Category::sortable()
            ->where('user_main_id', auth()->user()->main_id);
        
        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('position', 'like', '%' . $searchQuery . '%');
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.categories.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        return view('canteen.pages.categories.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        $request->validate([
            'name' => ['required', 'max:100', Rule::unique('categories')],
            'description' => ['nullable', 'string', 'max:255'],
            'position' => ['nullable', 'numeric'],
        ]);

        $category = new Category();
        $category->user_main_id = auth()->user()->main_id;
        $category->name = $request->name;
        $category->description = $request->description;
        $category->position = $request->position;
        $category->is_active = !empty($request->is_active) ? 1 : 0;
        $category->save();
        
        \Session::flash("message", "Category created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/categories'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $category = Category::where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        return view('canteen.pages.categories.edit', compact('category'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {

        //
        $request->validate([
            'name' => ['required', 'max:100', Rule::unique('categories')->ignore($id)],
            'description' => ['nullable', 'string', 'max:255'],
            'position' => ['nullable', 'numeric'],
        ]);

        $category = Category::where('user_main_id', auth()->user()->main_id)->find($id);
        $category->user_main_id = auth()->user()->main_id;
        $category->name = $request->name;
        $category->description = $request->description;
        $category->position = $request->position;
        $category->is_active = !empty($request->is_active) ? 1 : 0;
        $category->save();
        
        \Session::flash("message", "Category updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/categories'),
        ]);

    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {
        //
        $category = Category::where('user_main_id', auth()->user()->main_id)->findOrFail($id);
        $category->delete();

        \Session::flash("message", "Category deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/categories'),
        ]);

    }
}
