<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Item;
use App\Models\ItemAddon;
use App\Models\ItemBadge;
use App\Models\Tax;
use App\Models\Unit;
use App\Traits\FileUpload;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\DB;

class ItemController extends Controller implements HasMiddleware
{
    use FileUpload;

    public static function middleware(): array
    {
        return [
            new Middleware('permission:items.read', only: ['index']),
            new Middleware('permission:items.create', only: ['create', 'store']),
            new Middleware('permission:items.update', only: ['edit','update']),
            new Middleware('permission:items.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = Item::sortable()
            ->where('user_main_id', auth()->user()->main_id)
            ->whereNull('item_id');

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('position', 'like', '%' . $searchQuery . '%');
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.items.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $categories = Category::where('user_main_id', auth()->user()->main_id)
            ->where('is_active', 1)
            ->get();

        $taxes = Tax::where('user_main_id', auth()->user()->main_id)
            ->where('is_active', 1)
            ->get();

        $item_badges = ItemBadge::where('is_active', 1)
            ->get();

        $units = Unit::get();

        return view('canteen.pages.items.create', compact('categories', 'taxes', 'item_badges', 'units'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        //
        $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'description' => ['nullable', 'string'],
            'category_ids' => ['required', 'array'],
            'price' => ['required', 'numeric'],
            'stock_quantity' => ['nullable', 'numeric'],
            'image' => ['required', 'image', 'max:1024'],
            'ingredients' => ['nullable', 'string'],
            'allergen_info' => ['nullable', 'string'],
            'tax_ids' => ['nullable', 'array'],
            'badge_ids' => ['nullable', 'array'],
            'position' => ['nullable', 'numeric'],

            'addons.*.name' => ['required', 'string', 'max:100'],
            'addons.*.price' => ['required', 'numeric'],
            'addons.*.image' => ['nullable', 'image', 'max:1024'],

        ]);

        if($request->has_variant == "1"){
            $request->validate([
                'variants.*.name' => ['required', 'string', 'max:100'],
                'variants.*.price' => ['required', 'numeric'],
                'variants.*.stock_quantity' => ['nullable', 'numeric'],
            ]);
        }

        DB::transaction(function() use($request){

            $item = new Item();
            $item->user_main_id = auth()->user()->main_id;
            $item->name = $request->name;
            $item->description = $request->description;
            $item->category_ids = is_array($request->category_ids) ? array_map('intval', $request->category_ids) : [];
            $item->currency = auth()->user()->currency;
            $item->price = $request->price;
            $item->stock_quantity = $request->stock_quantity;
            if($request->has('image')){
                $imageURL = $this->uploadFile($request->image, 'items');
                if($imageURL['success'] == true){
                    $item->image = $imageURL['data'];
                }
            }
            $item->ingredients = $request->ingredients;
            $item->allergen_info = $request->allergen_info;
            $item->tax_ids = is_array($request->tax_ids) ? array_map('intval', $request->tax_ids) : [];
            $item->position = $request->position;
            $item->is_active = !empty($request->is_active) ? 1 : 0;
            $item->has_variant = $request->has_variant;
            $item->badge_ids = is_array($request->badge_ids) ? array_map('intval', $request->badge_ids) : [];
            $item->unit_id = $request->unit_id;

            if($request->filled('expiry_date')){
                $item->expiry_date = Carbon::parse($request->expiry_date)->toDateString();
            }

            if(count($item->tax_ids) > 0){
                $item->tax = $this->calculateTaxAmount($item->price, $item->tax_ids);
            }

            $item->save();
    
            if($request->has_variant == "1"){
                foreach ($request->variants as $variant) {
    
                    $variantItem = new Item();
                    $variantItem->item_id = $item->id;
    
                    $variantItem->name = $variant['name'];
                    $variantItem->price = $variant['price'];
                    $variantItem->stock_quantity = $variant['stock_quantity'];
                    $variantItem->expiry_date = (isset($variant['expiry_date']) && !empty($variant['expiry_date'])) ? Carbon::parse($variant['expiry_date'])->toDateString() : null;
    
                    $variantItem->user_main_id = $item->user_main_id;
                    $variantItem->description = $item->description;
                    $variantItem->category_ids = $item->category_ids;
                    $variantItem->currency = $item->currency;
                    $variantItem->image = $item->image;
                    $variantItem->ingredients = $item->ingredients;
                    $variantItem->allergen_info = $item->allergen_info;
                    $variantItem->tax_ids = $item->tax_ids;
                    $variantItem->position = $item->position;
                    $variantItem->is_active = $item->is_active;
                    $variantItem->has_variant = 0;
                    $variantItem->badge_ids = $item->badge_ids;

                    if(count($variantItem->tax_ids) > 0){
                        $variantItem->tax = $this->calculateTaxAmount($variantItem->price, $variantItem->tax_ids);
                    }

                    $variantItem->save();
    
                }
            }

            if(is_array($request->addons) && count($request->addons) > 0){

                foreach ($request->addons as $addon) {

                    $itemAddon = new ItemAddon();
                    $itemAddon->user_main_id = auth()->user()->main_id;
                    if(isset($addon['image'])){
                        $addonImageURL = $this->uploadFile($addon['image'], 'addons');
                        if($addonImageURL['success'] == true){
                            $itemAddon->image = $addonImageURL['data'];
                        }
                    }
                    $itemAddon->item_id = $item->id;
                    $itemAddon->name = $addon["name"];
                    $itemAddon->price = $addon["price"];
                    $itemAddon->is_required = !empty($addon["is_required"]) ? 1 : 0;
                    $itemAddon->save()  ;

                }

            }

        });

        \Session::flash("message", "Item created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/items'),
        ]);

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $item = Item::with('variants')->where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        $itemAddons = ItemAddon::where('item_id', $item->id)->get();

        $categories = Category::where('user_main_id', auth()->user()->main_id)
            ->where('is_active', 1)
            ->get();

        $taxes = Tax::where('user_main_id', auth()->user()->main_id)
            ->where('is_active', 1)
            ->get();

        $item_badges = ItemBadge::where('is_active', 1)
            ->get();

        $units = Unit::get();

        return view('canteen.pages.items.edit', compact('categories', 'taxes', 'item_badges', 'item', 'units', 'itemAddons'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {

        //
        $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'description' => ['nullable', 'string'],
            'category_ids' => ['required', 'array'],
            'price' => ['required', 'numeric'],
            'stock_quantity' => ['nullable', 'numeric'],
            'image' => ['nullable', 'image', 'max:1024'],
            'ingredients' => ['nullable', 'string'],
            'allergen_info' => ['nullable', 'string'],
            'tax_ids' => ['nullable', 'array'],
            'badge_ids' => ['nullable', 'array'],
            'position' => ['nullable', 'numeric'],

            'addons.*.name' => ['required', 'string', 'max:100'],
            'addons.*.price' => ['required', 'numeric'],
            'addons.*.image' => ['nullable', 'image', 'max:1024'],

        ]);

        if($request->has_variant == "1"){
            $request->validate([
                'variants.*.name' => ['required', 'string', 'max:100'],
                'variants.*.price' => ['required', 'numeric'],
                'variants.*.stock_quantity' => ['nullable', 'numeric'],
            ]);
        }

        $item = Item::where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        DB::transaction(function() use($item, $request){

            $item->user_main_id = auth()->user()->main_id;
            $item->name = $request->name;
            $item->description = $request->description;
            $item->category_ids = is_array($request->category_ids) ? array_map('intval', $request->category_ids) : [];
            $item->currency = auth()->user()->currency;
            $item->price = $request->price;
            $item->stock_quantity = $request->stock_quantity;
            if($request->has('image')){
                $imageURL = $this->uploadFile($request->image, 'items');
                if($imageURL['success'] == true){
                    $this->deleteFile($item->image);
                    $item->image = $imageURL['data'];
                }
            }
            $item->ingredients = $request->ingredients;
            $item->allergen_info = $request->allergen_info;
            $item->tax_ids = is_array($request->tax_ids) ? array_map('intval', $request->tax_ids) : [];
            $item->position = $request->position;
            $item->is_active = !empty($request->is_active) ? 1 : 0;
            // $item->has_variant = $request->has_variant;
            $item->badge_ids = is_array($request->badge_ids) ? array_map('intval', $request->badge_ids) : [];
            $item->unit_id = $request->unit_id;
            $item->expiry_date = Carbon::parse($request->expiry_date)->toDateString();

            if(count($item->tax_ids) > 0){
                $item->tax = $this->calculateTaxAmount($item->price, $item->tax_ids);
            }

            $item->save();
    
            if($item->has_variant == "1"){

                $existing_variant_ids = array_column($request->variants, 'id');
                $removed_variant_count = Item::where('item_id', $item->id)->whereNotIn('id', $existing_variant_ids)->count();

                if($removed_variant_count > 0){
                    Item::where('item_id', $item->id)->whereNotIn('id', $existing_variant_ids)->delete();
                }
    
                foreach ($request->variants as $variant) {
                    
                    if(isset($variant['id'])){
                        $variantItem = Item::find($variant['id']);
                    }else{
                        $variantItem = new Item();
                    }

                    $variantItem->item_id = $item->id;
    
                    $variantItem->name = $variant['name'];
                    $variantItem->price = $variant['price'];
                    $variantItem->stock_quantity = $variant['stock_quantity'];

                    if($request->filled('expiry_date')){
                        $variantItem->expiry_date = Carbon::parse($variant['expiry_date'])->toDateString();
                    }
    
                    $variantItem->user_main_id = $item->user_main_id;
                    $variantItem->description = $item->description;
                    $variantItem->category_ids = $item->category_ids;
                    $variantItem->currency = $item->currency;
                    $variantItem->image = $item->image;
                    $variantItem->ingredients = $item->ingredients;
                    $variantItem->allergen_info = $item->allergen_info;
                    $variantItem->tax_ids = $item->tax_ids;
                    $variantItem->position = $item->position;
                    $variantItem->is_active = $item->is_active;
                    $variantItem->has_variant = 0;
                    $variantItem->badge_ids = $item->badge_ids;

                    if(count($variantItem->tax_ids) > 0){
                        $variantItem->tax = $this->calculateTaxAmount($variantItem->price, $variantItem->tax_ids);
                    }

                    $variantItem->save();
    
                }
            }else{
                
                $variant_count = Item::where('item_id', $item->id)->count();
                
                if($variant_count > 0){
                    Item::where('item_id', $item->id)->delete();
                }

            }

            if(is_array($request->addons) && count($request->addons) > 0){

                $existing_addon_ids = array_column($request->addons, 'id');
                $removed_addon_count = ItemAddon::where('item_id', $item->id)->whereNotIn('id', $existing_addon_ids)->count();

                if($removed_addon_count > 0){
                    $remove_addons = ItemAddon::where('item_id', $item->id)->whereNotIn('id', $existing_addon_ids)->get();
                    foreach ($remove_addons as $remove_addon) {
                        $this->deleteFile($remove_addon->image);
                    }
                    $remove_addons->delete();
                }

                foreach ($request->addons as $addon) {

                    if(isset($addon['id'])){
                        $itemAddon = ItemAddon::find($addon['id']);
                    }else{
                        $itemAddon = new ItemAddon();
                    }

                    if(isset($addon['image'])){
                        $addonImageURL = $this->uploadFile($addon['image'], 'addons');
                        if($addonImageURL['success'] == true){
                            if(isset($itemAddon->id)){
                                $this->deleteFile($itemAddon->image);
                            }
                            $itemAddon->image = $addonImageURL['data'];
                        }
                    }

                    $itemAddon->user_main_id = auth()->user()->main_id;
                    $itemAddon->item_id = $item->id;
                    $itemAddon->name = $addon["name"];
                    $itemAddon->price = $addon["price"];
                    $itemAddon->is_required = !empty($addon["is_required"]) ? 1 : 0;
                    $itemAddon->save();

                }

            }else{

                $itemAddon_count = ItemAddon::where('item_id', $item->id)->count();
                if($itemAddon_count > 0){
                    $remove_addons = ItemAddon::where('item_id', $item->id)->get();
                    foreach ($remove_addons as $remove_addon) {
                        $this->deleteFile($remove_addon->image);
                    }
                    $remove_addons->delete();
                }

            }

        });

        \Session::flash("message", "Item updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/items'),
        ]);
    
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {
        //
        $item = Item::where('user_main_id', auth()->user()->main_id)->findOrFail($id);
        $item->delete();

        \Session::flash("message", "Item deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/items'),
        ]);
    }

    private function calculateTaxAmount($price, $tax_ids){

        $taxes = Tax::find($tax_ids);
        $tax_amount = 0;

        foreach ($taxes as $tax) {
            if($tax->type == "fixed"){
                $tax_amount += $tax->value;
            }else{
                $tax_amount += ($tax->value / 100) * $price;
            }
        }

        return $tax_amount;
    }

}
