<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Customer;
use App\Models\Item;
use App\Models\ItemAddon;
use App\Models\Order;
use App\Models\OrderStatus;
use App\Models\PaymentMethod;
use App\Models\PaymentStatus;
use App\Models\Tax;
use App\Models\User;
use App\Traits\OrderUtils;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use stdClass;

class OrderController extends Controller implements HasMiddleware
{
    use OrderUtils;

    public static function middleware(): array
    {
        return [
            new Middleware('permission:orders.read', only: ['index']),
            new Middleware('permission:orders.update', only: ['edit','update']),
        ];
    }

    public function index(Request $request){

        //
        $rowsQuery = Order::sortable()
            ->where('user_main_id', auth()->user()->main_id);

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('net_amount', 'like', '%' . $searchQuery . '%')
                ->orWhere('payable_amount', 'like', '%' . $searchQuery . '%')
                ->orWhereHas('customer', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%')
                    ->orWhere('email', 'like', '%' . $searchQuery . '%')
                    ->orWhere('mobile_number', 'like', '%' . $searchQuery . '%');
                })
                ->orWhereHas('order_status', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%');
                })
                ->orWhereHas('payment_method', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%');
                })
                ->orWhereHas('payment_status', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%');
                });
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.orders.index', compact('rows'));
    }

    public function orderReceipt($orderId){
        
        return view('canteen.pages.pos.order-invoice');

    }

    public function edit(string $subdomain, string $id){

        $canteen = User::find(auth()->user()->main_id);

        $order = Order::with([
                'customer',
                'order_items' => function($query){
                    $query->withCount('main_item_addon');
                },
                'order_items.item_addons',
                'order_items.item_taxes',
                'order_items.item',
            ])
            ->where('user_main_id', $canteen->main_id)->findOrFail($id);

        $categories = Category::whereHas('items', function($q){
                        $q->where('is_active', 1);
                    })->where('user_main_id', auth()->user()->main_id)
                    ->where('is_active', 1)
                    ->get();
        
        $items = Item::with('main_item', 'categories')
                ->whereHas('categories', function($q){
                    $q->where('is_active', 1);
                })
                ->whereNull('item_id')
                ->where('user_main_id', auth()->user()->main_id)
                ->get()
                ->map(function($item){
                    $item['addon_count'] = 0;
                    if(!empty($item->item_id)){
                        $item['addon_count'] = ItemAddon::where('item_id', $item->item_id)->count();
                    }else{
                        $item['addon_count'] = ItemAddon::where('item_id', $item->id)->count();
                    }
                    return $item;
                });
        
        $taxes = Tax::where('user_main_id', auth()->user()->main_id)
                ->where('is_active', 1)
                ->get()
                ->keyBy('id');

        // $customers = Customer::select('id', 'name')->where('user_main_id', auth()->user()->main_id)->get();
        $payment_methods = PaymentMethod::find(is_array($canteen->pos_payment_methods_ids) ? $canteen->pos_payment_methods_ids : []);
        $payment_statuses = PaymentStatus::get();
        $order_statuses = OrderStatus::get();
        
        return view('canteen.pages.orders.edit', compact('canteen', 'categories', 'items', 'taxes', 'payment_methods', 'payment_statuses', 'order_statuses', 'order'));

    }

    public function update(Request $request, string $subdomain, string $id){

        $request->validate([
            'restaurant_instruction' => ['nullable', 'string'],
            'payment_method_id' => ['required', 'numeric'],
            'payment_status_id' => ['required', 'numeric'],
            'order_status_id' => ['required', 'numeric'],
            'cartItem' => ['required', 'array'],
            'cartItem.*.item_id' => ['required', 'numeric'],
            'cartItem.*.quantity' => ['required', 'numeric'],
            'cartItem.*.discount' => ['required', 'numeric'],
        ]);
        
        $order = Order::where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        if($order->order_status_id == 5){
            return response()->json([
                'success' => false,
                'message' => 'Canceled orders cannot be updated.'
            ]);
        }

        if($order->payment_status_id == 2){

            if($order->order_status_id != $request->order_status_id){

                if (auth()->user()->can('orders.update-status')) {
                    $order->order_status_id = $request->order_status_id;
                }
                $order->save();

                \Session::flash("message", "Order status updated successfully");

                return response()->json([
                    'success' => true,
                    'redirect_url' => url('/app/orders'),
                ]);

            }

            return response()->json([
                'success' => false,
                'message' => 'Once an order is paid, only the order status can be changed.'
            ]);
        }

        $reqOrderObj = new stdClass();
        $reqOrderObj->order_id = $order->id;
        $reqOrderObj->restaurant_instruction = $request->restaurant_instruction;
        $reqOrderObj->payment_method_id = $request->payment_method_id;
        $reqOrderObj->payment_status_id = $request->payment_status_id;
        $reqOrderObj->payment_status_id = $request->payment_status_id;
        if (auth()->user()->can('orders.update-status')) {
            $reqOrderObj->order_status_id = $request->order_status_id;
        }else{
            $reqOrderObj->order_status_id = $order->order_status_id;
        }
        $reqOrderObj->cartItem = $request->cartItem;
        $reqOrderObj->need_cutlery = $request->need_cutlery;

        $response = $this->ou_update_order($reqOrderObj);

        if($response["success"]){
                
            \Session::flash("message", "Order updated successfully");

            return response()->json([
                'success' => true,
                'redirect_url' => url('/app/orders'),
            ]);

        }

    }

}
