<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use App\Models\User;
use App\Traits\FileUpload;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller implements HasMiddleware
{
    use FileUpload;

    public static function middleware(): array
    {
        return [
            new Middleware('permission:others.business-setting', only: ['businessSettingForm', 'businessSetting'])
        ];
    }

    public function myAccountForm(){
        
        if(auth()->user()->id == auth()->user()->main_id){
            return redirect('/app');
        }

        $canteen = User::find(auth()->user()->main_id);
        $user = User::find(auth()->user()->id);

        return view('canteen.pages.profile.myAccount', compact('canteen', 'user'));
    }

    public function myAccountUpdate(Request $request){

        $request->merge(['mobile_number' => str_replace([' ', '-'],'', $request->mobile_number)]);

        $request->validate([
            'name' => ['required', 'max:100'],
            'email' => ['required', 'email:rfc,dns', Rule::unique('users', 'email')->where(function ($query){
                return $query->whereNull('deleted_at');          
            })->ignore(auth()->user()->id)],
            'mobile_number' => ['required', Rule::unique('users', 'mobile_number')->where(function ($query){
                return $query->whereNull('deleted_at');          
            })->ignore(auth()->user()->id)],
            'address' => ['required', 'max:160']
        ]);

        $user = User::find(auth()->user()->id);
        $user->name = $request->name;
        $user->email = $request->email;
        $user->mobile_number = $request->mobile_number;
        $user->address = $request->address;
        $user->save();

        \Session::flash("message", "Account Settings updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/my-account'),
        ]);

    }

    public function businessSettingForm(){

        // $this->checkPermission();

        $business = User::find(auth()->user()->main_id);
        $payment_methods = PaymentMethod::where('is_active', 1)->get();

        return view('canteen.pages.profile.businessSettings', compact('business', 'payment_methods'));
    }

    public function businessSetting(Request $request){

        // $this->checkPermission();

        $request->merge(['business_mobile_number' => str_replace([' ', '-'],'', $request->business_mobile_number)]);

        $request->validate([
            'logo' => ['nullable', 'image', 'max:512'],
            'business_name' => ['required', 'max:100'],
            'business_email' => ['required', 'email:rfc,dns', Rule::unique('users', 'email')->where(function ($query){
                return $query->whereNull('deleted_at');          
            })->ignore(auth()->user()->main_id)],
            'business_mobile_number' => ['required', Rule::unique('users', 'mobile_number')->where(function ($query){
                return $query->whereNull('deleted_at');          
            })->ignore(auth()->user()->main_id)],
            'user_name' => ['required', 'max:100', Rule::unique('users', 'user_name')->where(function ($query){
                return $query->whereNull('deleted_at');          
            })->ignore(auth()->user()->main_id)],
            'owner_name' => ['required', 'max:100'],
            'business_address' => ['required', 'max:160'],
            'pos_payment_methods_ids' => ['required', 'array'],
            'web_payment_methods_ids' => ['required', 'array'],
        ]);

        $user = User::find(auth()->user()->main_id);

        if($request->hasFile('logo')){
            $logoURL = $this->uploadFile($request->logo, 'logos');
            if($logoURL['success'] == true){
                $this->deleteFile($user->logo);
                $user->logo = $logoURL['data'];
            }
        }

        $user->business_name = $request->business_name;
        $user->email = $request->business_email;
        $user->mobile_number = $request->business_mobile_number;
        $user->user_name = $request->user_name;
        $user->email = $request->business_email;
        $user->pos_payment_methods_ids = is_array($request->pos_payment_methods_ids) ? array_map('intval', $request->pos_payment_methods_ids) : [];
        $user->web_payment_methods_ids = is_array($request->web_payment_methods_ids) ? array_map('intval', $request->web_payment_methods_ids) : [];
        $user->address = $request->business_address;
        $user->is_available = !empty($request->is_available) ? 1 : 0;
        $user->save();

        \Session::flash("message", "Business Settings updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/business-settings'),
        ]);

    }

    public function changePassword(Request $request){

        $request->validate([
            'current_password' => ['required', 'current_password'],
            'new_password' => [
                'required',
                'confirmed',
                Password::min(6)->letters()->mixedCase()->numbers()->symbols()
            ],
        ]);

        $user = User::find(auth()->user()->id);
        $user->password = Hash::make($request->new_password);
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Your password has been changed.',
        ]);

    }

    private function checkPermission(){

        // dd(auth()->user()->id, auth()->user()->main_id);
        $user = auth()->user();

        if (!($user->can('others.business-setting') || $user->id == $user->main_id)) {
            abort(403, 'Unauthorized action.');
        }

    }

}
