<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Hash;

class TableController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:tables.read', only: ['index']),
            new Middleware('permission:tables.create', only: ['create', 'store']),
            new Middleware('permission:tables.update', only: ['edit','update']),
            new Middleware('permission:tables.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = Customer::sortable()
            ->where('type', 'table')
            ->where('user_main_id', auth()->user()->main_id);

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('address', 'like', '%' . $searchQuery . '%');
            });

        }
        
        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.tables.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $employees = User::sortable()->where('type', 'employee')
            ->where('main_id', auth()->user()->main_id)
            ->select('id', 'name')
            ->get();


        return view('canteen.pages.tables.create', compact('employees'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        //
        $request->validate([
            'name' => ['required', 'max:100'],
            'remark' => ['nullable', 'max:120'],
            'employee_id' => ['nullable', 'numeric']
        ]);

        $password = "Pass@". rand(1000, 9999);

        $table = new Customer();
        $table->user_main_id = auth()->user()->main_id;

        $table->name = $request->name;
        $table->address = $request->remark;

        $table->email = uniqid().'@table.com';
        $table->mobile_number = uniqid();
        $table->otp_verified_at = Carbon::now()->toDateString();
        $table->password = Hash::make($password);
        $table->is_verified = 1;
        $table->country = "BD";
        $table->type = "table";

        $table->employee_id = $request->employee_id;
        $table->save();
        
        \Session::flash("message", "Table created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/tables'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $table = Customer::where('user_main_id', auth()->user()->main_id)->where('type', 'table')->findOrFail($id);

        $employees = User::sortable()->where('type', 'employee')
            ->where('main_id', auth()->user()->main_id)
            ->select('id', 'name')
            ->get();

        return view('canteen.pages.tables.edit', compact('table', 'employees'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {

        //
        $request->validate([
            'name' => ['required', 'max:100'],
            'remark' => ['nullable', 'max:120'],
            'employee_id' => ['nullable', 'numeric']
        ]);

        $table = Customer::where('user_main_id', auth()->user()->main_id)->where('type', 'table')->find($id);
        $table->name = $request->name;
        $table->address = $request->remark;
        $table->employee_id = $request->employee_id;

        $table->save();
        
        \Session::flash("message", "Table updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/tables'),
        ]);
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {
        //
        $table = Customer::where('user_main_id', auth()->user()->main_id)->where('type', 'table')->findOrFail($id);
        $table->delete();

        \Session::flash("message", "Table deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/tables'),
        ]);
    }
}
