<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Tax;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class TaxController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:taxes.read', only: ['index']),
            new Middleware('permission:taxes.create', only: ['create', 'store']),
            new Middleware('permission:taxes.update', only: ['edit','update']),
            new Middleware('permission:taxes.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = Tax::sortable()
            ->with('user:id,currency')
            ->where('user_main_id', auth()->user()->main_id);
        
        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('value', 'like', '%' . $searchQuery . '%')
                ->orWhere('position', 'like', '%' . $searchQuery . '%');
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.taxes.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        return view('canteen.pages.taxes.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'max:100'],
            'type' => ['required', 'max:32'],
            'value' => ['required', 'numeric'],
            'position' => ['nullable', 'numeric'],
        ]);

        $tax = new Tax();
        $tax->user_main_id = auth()->user()->main_id;
        $tax->name = $request->name;
        $tax->type = $request->type;
        $tax->value = $request->value;
        $tax->position = $request->position;
        $tax->is_active = !empty($request->is_active) ? 1 : 0;
        $tax->save();
        
        \Session::flash("message", "Tax created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/taxes'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {   
        $tax = Tax::where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        return view('canteen.pages.taxes.edit', compact('tax'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {
        $request->validate([
            'name' => ['required', 'max:100'],
            'type' => ['required', 'max:32'],
            'value' => ['required', 'numeric'],
            'position' => ['nullable', 'numeric'],
        ]);

        $tax = Tax::where('user_main_id', auth()->user()->main_id)->find($id);
        $tax->user_main_id = auth()->user()->main_id;
        $tax->name = $request->name;
        $tax->type = $request->type;
        $tax->value = $request->value;
        $tax->position = $request->position;
        $tax->is_active = !empty($request->is_active) ? 1 : 0;
        $tax->save();
        
        \Session::flash("message", "Tax updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/taxes'),
        ]);
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {

        $tax = Tax::where('user_main_id', auth()->user()->main_id)->findOrFail($id);
        $tax->delete();

        \Session::flash("message", "Tax deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/taxes'),
        ]);

    }



}
